// Background script for HackyChat extension
import config from './config.js'
const API_BASE_URL = config.apiUrl

// Listen for tab updates to check for existing rooms
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && tab.url) {
    await checkForRoom(tab.url, tabId)
  }
})

// Listen for messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'checkRoom') {
    checkForRoom(request.url, sender.tab?.id)
    sendResponse({ success: true })
  } else if (request.action === 'openSidePanel') {
    chrome.sidePanel.open({ tabId: sender.tab?.id })
    sendResponse({ success: true })
  } else if (request.action === 'getCurrentTabId') {
    // Return the tab ID from the sender
    sendResponse({ tabId: sender.tab?.id || null })
  }
  return true // Keep message channel open for async response
})

async function checkForRoom(url, tabId) {
  try {
    const token = await getStoredToken()
    if (!token) {
      // No token, show create account option
      updateBadge('?', tabId)
      return
    }

    const response = await fetch(`${API_BASE_URL}/rooms/search?url=${encodeURIComponent(url)}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      if (data.room) {
        updateBadge('💬', tabId)
        await chrome.storage.local.set({ [`room_${tabId}`]: data.room })
      } else {
        updateBadge('+', tabId)
        await chrome.storage.local.remove([`room_${tabId}`])
      }
    } else {
      updateBadge('!', tabId)
    }
  } catch (error) {
    console.error('Error checking for room:', error)
    updateBadge('!', tabId)
  }
}

function updateBadge(text, tabId) {
  chrome.action.setBadgeText({ text, tabId })
  chrome.action.setBadgeBackgroundColor({ color: '#3B82F6', tabId })
}

async function getStoredToken() {
  const result = await chrome.storage.local.get(['hackychat_token'])
  return result.hackychat_token
}

async function setStoredToken(token) {
  await chrome.storage.local.set({ hackychat_token: token })
}

// Handle authentication
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'authenticate') {
    authenticateUser().then(sendResponse)
    return true // Keep message channel open for async response
  }
})

async function authenticateUser() {
  try {
    const response = await fetch(`${API_BASE_URL}/users/anonymous`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      await setStoredToken(data.token)
      return { success: true, user: data.user, token: data.token }
    } else {
      throw new Error('Failed to create anonymous user')
    }
  } catch (error) {
    console.error('Authentication error:', error)
    return { success: false, error: error.message }
  }
}
