// Content script for HackyChat extension
// No need for API_BASE_URL here - all API calls go through background.js

// Function to dismiss the indicator with animation
function dismissIndicator(indicator) {
  indicator.style.opacity = '0'
  indicator.style.transition = 'opacity 0.3s ease-out, transform 0.3s ease-out'
  indicator.style.transform = 'translateX(100%)'
  setTimeout(() => {
    if (indicator.parentNode) {
      indicator.remove()
    }
  }, 300)
}

// Inject chat indicator into the page
function injectChatIndicator(room) {
  // Remove existing indicator if any
  const existingIndicator = document.getElementById('hackychat-indicator')
  if (existingIndicator) {
    existingIndicator.remove()
  }

  const indicator = document.createElement('div')
  indicator.id = 'hackychat-indicator'
  indicator.innerHTML = `
    <div style="
      position: fixed;
      top: 20px;
      right: 20px;
      background: #3B82F6;
      color: white;
      padding: 12px 16px;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
      z-index: 10000;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 14px;
      cursor: pointer;
      max-width: 300px;
      position: relative;
    ">
      <button id="hackychat-close" style="
        position: absolute;
        top: 4px;
        right: 4px;
        background: none;
        border: none;
        color: white;
        font-size: 16px;
        font-weight: bold;
        cursor: pointer;
        padding: 0;
        width: 20px;
        height: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0.7;
        border-radius: 2px;
      " title="Close">×</button>
      <div style="display: flex; align-items: center; gap: 8px; padding-right: 20px;">
        <span>💬</span>
        <div>
          <div style="font-weight: 600;">Chat Available!</div>
          <div style="font-size: 12px; opacity: 0.9;">${room.participant_count} people discussing this</div>
        </div>
      </div>
    </div>
  `

  // Add click handler for the main indicator (opens side panel)
  indicator.addEventListener('click', (e) => {
    // Don't open side panel if close button was clicked
    if (e.target.id === 'hackychat-close') {
      return
    }
    chrome.runtime.sendMessage({ action: 'openSidePanel' })
  })

  // Add click handler for the close button
  const closeButton = indicator.querySelector('#hackychat-close')
  closeButton.addEventListener('click', (e) => {
    e.stopPropagation() // Prevent the main click handler from firing
    dismissIndicator(indicator)
  })

  // Add hover effects for the close button
  closeButton.addEventListener('mouseenter', () => {
    closeButton.style.opacity = '1'
    closeButton.style.backgroundColor = 'rgba(255, 255, 255, 0.2)'
  })
  closeButton.addEventListener('mouseleave', () => {
    closeButton.style.opacity = '0.7'
    closeButton.style.backgroundColor = 'none'
  })

  document.body.appendChild(indicator)

  // Auto-hide after 5 seconds
  setTimeout(() => {
    if (indicator.parentNode) {
      dismissIndicator(indicator)
    }
  }, 5000)
}

// Get current tab ID - content scripts can query tabs directly
// This is simpler and more reliable than using devtools API
async function getCurrentTabId() {
  return new Promise((resolve) => {
    // First try to get it from the message sender context
    chrome.runtime.sendMessage({ action: 'getCurrentTabId' }, (response) => {
      if (chrome.runtime.lastError || !response?.tabId) {
        // Fallback: query tabs directly (requires tabs permission, which we have)
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          resolve(tabs[0]?.id || null)
        })
      } else {
        resolve(response.tabId)
      }
    })
  })
}

// Check for existing room when page loads
async function checkForRoom() {
  try {
    // Request room check from background script
    const response = await chrome.runtime.sendMessage({ action: 'checkRoom', url: window.location.href })

    // Get the current tab ID properly
    const tabId = await getCurrentTabId()
    if (!tabId) {
      console.warn('Could not get tab ID, room indicator may not work correctly')
      return
    }

    // Check if we have room data in storage for this tab
    const result = await chrome.storage.local.get([`room_${tabId}`])
    const room = result[`room_${tabId}`]

    console.log('Content script checkForRoom:', { tabId, room: !!room, url: window.location.href })

    if (room) {
      console.log('Injecting chat indicator for room:', room.title)
      injectChatIndicator(room)
    } else {
      console.log('No room found for this page')
    }
  } catch (error) {
    console.error('Error checking for room:', error)
  }
}

// Run when page loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', checkForRoom)
} else {
  checkForRoom()
}

// Listen for messages from popup/background
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'injectTestNotification' && request.room) {
    console.log('Injecting test notification:', request.room)
    injectChatIndicator(request.room)
    sendResponse({ success: true })
  }
  return true
})

// Listen for storage changes (when room data is updated)
chrome.storage.onChanged.addListener(async (changes, namespace) => {
  if (namespace === 'local') {
    // Get current tab ID to check if this change is for our tab
    const tabId = await getCurrentTabId()
    if (!tabId) return

    const roomKey = `room_${tabId}`
    if (changes[roomKey] && changes[roomKey].newValue) {
      console.log('Storage change detected, injecting indicator for:', changes[roomKey].newValue.title)
      injectChatIndicator(changes[roomKey].newValue)
    }
  }
})
