#!/bin/bash

# HackyChat Extension Packaging Script
# Prepares extension for Chrome Web Store and Firefox Add-ons submission

set -e

echo "🚀 Packaging HackyChat Extension for Store Submission"

# Get version from manifest
VERSION=$(grep '"version"' manifest.json | cut -d'"' -f4)
OUTPUT_FILE="hackyChat-extension-v${VERSION}.zip"

echo "📦 Version: $VERSION"
echo "📁 Output: $OUTPUT_FILE"

# Check for required files
echo "🔍 Checking required files..."

REQUIRED_FILES=(
    "manifest.json"
    "popup.html"
    "popup.js"
    "sidepanel.html"
    "sidepanel.js"
    "content-script.js"
    "background.js"
    "config.js"
    "PRIVACY_POLICY.md"
    "TERMS_OF_SERVICE.md"
    "README.md"
)

for file in "${REQUIRED_FILES[@]}"; do
    if [ ! -f "$file" ]; then
        echo "❌ Missing required file: $file"
        exit 1
    fi
done

# Check for icon files
ICON_SIZES=("16" "32" "48" "128")
for size in "${ICON_SIZES[@]}"; do
    if [ ! -f "icons/icon${size}.png" ]; then
        echo "❌ Missing icon: icons/icon${size}.png"
        exit 1
    fi
done

echo "✅ All required files present"

# Validate manifest.json
echo "🔍 Validating manifest.json..."
if ! jq empty manifest.json 2>/dev/null; then
    echo "❌ Invalid JSON in manifest.json"
    exit 1
fi

echo "✅ Manifest is valid JSON"

# Create ZIP file (excluding development files)
echo "📦 Creating ZIP package..."

# Files to exclude
EXCLUDE_PATTERNS=(
    "*.git*"
    "*.DS_Store"
    "*node_modules*"
    "*.log"
    "*package-extension.sh"
    "*README.dev.md"
)

# Build exclude args for zip
EXCLUDE_ARGS=""
for pattern in "${EXCLUDE_PATTERNS[@]}"; do
    EXCLUDE_ARGS="$EXCLUDE_ARGS -x $pattern"
done

# Create ZIP
eval "zip -r $OUTPUT_FILE . $EXCLUDE_ARGS"

echo "✅ Package created: $OUTPUT_FILE"

# Show package contents
echo "📋 Package contents:"
unzip -l "$OUTPUT_FILE" | head -20

echo ""
echo "🎉 Extension packaged successfully!"
echo ""
echo "📤 Ready for store submission:"
echo "   Chrome Web Store: https://chrome.google.com/webstore/developer/dashboard"
echo "   Firefox Add-ons: https://addons.mozilla.org/developers/"
echo ""
echo "📋 Next steps:"
echo "   1. Test the extension thoroughly"
echo "   2. Create store listing screenshots"
echo "   3. Submit to both stores"
echo "   4. Set up auto-update when approved"
echo ""
echo "🔗 Store Listing Details:"
echo "   - Title: HackyChat"
echo "   - Description: Join conversations about the stories you're reading"
echo "   - Privacy Policy: Include PRIVACY_POLICY.md content"
echo "   - Category: Social & Communication"
