// Popup script for HackyChat extension
import config from './config.js'

const API_BASE_URL = config.apiUrl
const FRONTEND_URL = config.frontendUrl

let currentUser = null
let currentRoom = null

document.addEventListener('DOMContentLoaded', async () => {
  await initializePopup()
  setupEventListeners()
})

async function initializePopup() {
  try {
    // Check if user is authenticated
    const token = await getStoredToken()
    if (token) {
      await loadUserInfo(token)
    } else {
      await authenticateUser()
    }

    // Check for room on current page
    await checkCurrentPageRoom()
  } catch (error) {
    console.error('Initialization error:', error)
    updateStatus('Error connecting to HackyChat', 'disconnected')
  }
}

async function getStoredToken() {
  const result = await chrome.storage.local.get(['hackychat_token'])
  return result.hackychat_token
}

async function authenticateUser() {
  try {
    const response = await fetch(`${API_BASE_URL}/users/anonymous`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      await chrome.storage.local.set({ hackychat_token: data.token })
      currentUser = data.user
      updateUserInfo(data.user)
      updatePrivacySettings(data.user)
      updateStatus('Connected to HackyChat', 'connected')
    } else {
      throw new Error('Failed to authenticate')
    }
  } catch (error) {
    console.error('Authentication error:', error)
    updateStatus('Failed to connect', 'disconnected')
  }
}

async function loadUserInfo(token) {
  try {
    const response = await fetch(`${API_BASE_URL}/users/me`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      currentUser = data.user
      updateUserInfo(data.user)
      updatePrivacySettings(data.user)
      updateStatus('Connected to HackyChat', 'connected')
    } else {
      // Token is invalid, re-authenticate
      await chrome.storage.local.remove(['hackychat_token'])
      await authenticateUser()
    }
  } catch (error) {
    console.error('Error loading user info:', error)
    updateStatus('Connection error', 'disconnected')
  }
}

async function checkCurrentPageRoom() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    const response = await fetch(`${API_BASE_URL}/rooms/search?url=${encodeURIComponent(tab.url)}`, {
      headers: {
        'Authorization': `Bearer ${await getStoredToken()}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      if (data.room) {
        currentRoom = data.room
        showRoomActions()
      } else {
        showCreateRoomAction()
      }
    }
  } catch (error) {
    console.error('Error checking for room:', error)
  }
}

function updateStatus(message, type) {
  const statusEl = document.getElementById('status')
  statusEl.textContent = message
  statusEl.className = `status ${type}`
}

function updateUserInfo(user) {
  const userInfoEl = document.getElementById('user-info')
  const usernameEl = document.getElementById('username')
  usernameEl.textContent = user.username
  userInfoEl.style.display = 'block'
}

function updatePrivacySettings(user) {
  const settingsEl = document.getElementById('privacy-settings')
  const optOutEl = document.getElementById('data-opt-out')
  if (!settingsEl || !optOutEl) return
  optOutEl.checked = !!user.data_collection_opt_out
  settingsEl.style.display = 'block'
}

function showRoomActions() {
  document.getElementById('open-chat').style.display = 'block'
  document.getElementById('create-room').style.display = 'none'
}

function showCreateRoomAction() {
  document.getElementById('open-chat').style.display = 'none'
  document.getElementById('create-room').style.display = 'block'
}

function setupEventListeners() {
  const optOutEl = document.getElementById('data-opt-out')
  if (optOutEl) {
    optOutEl.addEventListener('change', async (event) => {
      const token = await getStoredToken()
      if (!token) return

      try {
        const response = await fetch(`${API_BASE_URL}/users/profile`, {
          method: 'PATCH',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({
            user: {
              data_collection_opt_out: event.target.checked
            }
          })
        })

        if (response.ok) {
          const data = await response.json()
          currentUser = data.user
        }
      } catch (error) {
        console.error('Failed to update privacy setting:', error)
      }
    })
  }

  document.getElementById('open-chat').addEventListener('click', async () => {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
      if (tab && tab.id) {
        await chrome.sidePanel.open({ tabId: tab.id })
      } else {
        console.error('Could not get current tab')
      }
    } catch (error) {
      console.error('Error opening side panel:', error)
    }
  })

  document.getElementById('create-room').addEventListener('click', async () => {
    await createRoomForCurrentPage()
  })

  document.getElementById('view-trending').addEventListener('click', () => {
    chrome.tabs.create({ url: FRONTEND_URL })
  })

  // Debug info (only show if shift+click on view trending)
  let debugClicks = 0
  document.getElementById('view-trending').addEventListener('click', (e) => {
    if (e.shiftKey) {
      e.preventDefault()
      debugClicks++
      if (debugClicks >= 3) {
        showDebugInfo()
        debugClicks = 0
      }
    }
  })

  // Test notification button (only in debug mode)
  const testButton = document.createElement('button')
  testButton.id = 'test-notification'
  testButton.className = 'button secondary'
  testButton.style.display = 'none'
  testButton.textContent = 'Test Notification'
  testButton.addEventListener('click', async () => {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    const testRoom = {
      id: 999,
      title: 'Test Room',
      participant_count: 5,
      canonical_url: tab.url || 'https://example.com'
    }
    chrome.tabs.sendMessage(tab.id, { action: 'injectTestNotification', room: testRoom })
  })
  document.body.insertBefore(testButton, document.getElementById('debug-info'))
}

function showDebugInfo() {
  const debugEl = document.getElementById('debug-info')
  const debugContent = document.getElementById('debug-content')

  // Show test button
  document.getElementById('test-notification').style.display = 'block'

  // Gather debug info
  chrome.storage.local.get(['hackychat_token'], async (result) => {
    const token = result.hackychat_token
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    debugContent.innerHTML = `
      <div>Token: ${token ? 'Present' : 'None'}</div>
      <div>Current URL: ${tab?.url || 'Unknown'}</div>
      <div>Tab ID: ${tab?.id || 'Unknown'}</div>
      <div>Extension Version: ${chrome.runtime.getManifest().version}</div>
    `

    debugEl.style.display = 'block'
  })
}

async function createRoomForCurrentPage() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    
    const response = await fetch(`${API_BASE_URL}/rooms`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${await getStoredToken()}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        url: tab.url,
        title: tab.title
      })
    })

    if (response.ok) {
      const data = await response.json()
      currentRoom = data.room
      showRoomActions()
      updateStatus('Room created!', 'connected')
      
      // Open side panel to show the new room
      setTimeout(() => {
        chrome.sidePanel.open()
      }, 1000)
    } else {
      updateStatus('Failed to create room', 'disconnected')
    }
  } catch (error) {
    console.error('Error creating room:', error)
    updateStatus('Error creating room', 'disconnected')
  }
}
