// Sidepanel script for HackyChat extension
import config from './config.js'

const API_BASE_URL = config.apiUrl
const WS_URL = config.wsUrl

let currentUser = null
let currentRoom = null
let ws = null
let reconnectTimeout = null
let reconnectAttempts = 0
let shouldReconnect = true
let isSubscribed = false
let fallbackPolling = null
let useFallback = false
let lastMessageId = null

// Exponential backoff: 1s, 2s, 4s, 8s, 16s, max 30s with jitter
function getReconnectDelay(attempts) {
  const baseDelay = 1000 // 1 second
  const exponentialDelay = Math.min(baseDelay * Math.pow(2, attempts), 30000) // Cap at 30s
  // Add jitter (±20%) to prevent thundering herd
  const jitter = exponentialDelay * 0.2 * (Math.random() * 2 - 1)
  return exponentialDelay + jitter
}

document.addEventListener('DOMContentLoaded', async () => {
  await initializeSidepanel()
  setupEventListeners()
})

async function fetchCurrentUser() {
  try {
    const token = await getStoredToken()
    if (!token) return null

    const response = await fetch(`${API_BASE_URL}/users/me`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      currentUser = data.user
      updateUserDisplay()
      return currentUser
    }
  } catch (error) {
    console.error('Error fetching user:', error)
  }
  return null
}

function updateUserDisplay() {
  const roomInfoEl = document.getElementById('room-info')
  const statusEl = document.getElementById('connection-status')
  if (currentUser && roomInfoEl && currentRoom) {
    roomInfoEl.textContent = `${currentRoom.title} • ${currentRoom.participant_count} participants • You: ${currentUser.username}`
  } else if (roomInfoEl && currentRoom) {
    roomInfoEl.textContent = `${currentRoom.title} • ${currentRoom.participant_count} participants`
  }
  if (statusEl) {
    const wsReady = ws && ws.readyState === WebSocket.OPEN
    statusEl.textContent = isSubscribed
      ? 'Real-time connected'
      : (useFallback ? 'Using fallback (polling)…' : (wsReady ? 'Subscribing…' : 'Connecting…'))
  }
}

async function initializeSidepanel() {
  try {
    // Fetch current user first
    await fetchCurrentUser()

    // Get current tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })

    // Check for room on current page
    let token = await getStoredToken()
    if (!token) {
      console.log('No token found, attempting to authenticate...')
      const authResult = await chrome.runtime.sendMessage({ action: 'authenticate' })
      if (authResult && authResult.success) {
        token = authResult.token
      } else {
        showError('Authentication failed')
        return
      }
    }

    const response = await fetch(`${API_BASE_URL}/rooms/search?url=${encodeURIComponent(tab.url)}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      console.log('Room search response:', data)
      if (data.room) {
        currentRoom = data.room
        console.log('Current room set to:', currentRoom)
        updateUserDisplay()
        await loadRoom()
      } else {
        console.log('No room found, showing no room')
        showNoRoom()
      }
    } else {
      console.error('Room search failed:', response.status, response.statusText)
      showError('Failed to check for room')
    }
  } catch (error) {
    console.error('Initialization error:', error)
    showError(`Connection error: ${error.message}`)
  }
}

async function getStoredToken() {
  const result = await chrome.storage.local.get(['hackychat_token'])
  return result.hackychat_token
}

async function loadRoom() {
  try {
    updateRoomInfo()

    console.log('Loading room:', currentRoom)
    console.log('Is member:', currentRoom.is_member)
    console.log('Is member type:', typeof currentRoom.is_member)

    // Ensure we always check membership explicitly
    // Handle both boolean and undefined/null cases
    const isMember = currentRoom.is_member === true || currentRoom.is_member === 'true'

    console.log('Processed isMember:', isMember)

    // Check if user is a member
    if (isMember) {
      console.log('User is member, showing chat')
      shouldReconnect = true
      showChat()
      await loadMessages()
      connectWebSocket()
    } else {
      console.log('User is NOT a member, showing join prompt')
      // Ensure we don't try to connect WebSocket if not a member
      shouldReconnect = false
      if (ws) {
        try {
          ws.close()
        } catch (e) {
          console.error('Error closing WebSocket:', e)
        }
        ws = null
      }
      showJoinPrompt()
    }
  } catch (error) {
    console.error('Error loading room:', error)
    showError('Failed to load room')
  }
}

async function loadMessages() {
  try {
    const token = await getStoredToken()
    const response = await fetch(`${API_BASE_URL}/rooms/${currentRoom.id}/messages`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      const data = await response.json()
      const messages = data.messages || []
      if (messages.length > 0) {
        lastMessageId = messages[0].id // Track most recent message ID for polling
      }
      displayMessages(messages)
    }
  } catch (error) {
    console.error('Error loading messages:', error)
  }
}

// Fallback polling when WebSocket is unavailable
function startFallbackPolling() {
  if (fallbackPolling) {
    clearInterval(fallbackPolling)
  }

  console.log('📡 Starting fallback polling mode (every 3 seconds)')

  fallbackPolling = setInterval(async () => {
    if (!currentRoom || !currentRoom.is_member) {
      stopFallbackPolling()
      return
    }

    try {
      const token = await getStoredToken()
      if (!token) return

      // Fetch new messages since lastMessageId
      const url = lastMessageId
        ? `${API_BASE_URL}/rooms/${currentRoom.id}/messages?since=${lastMessageId}`
        : `${API_BASE_URL}/rooms/${currentRoom.id}/messages`

      const response = await fetch(url, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      })

      if (response.ok) {
        const data = await response.json()
        const messages = data.messages || []

        // Add any new messages
        messages.forEach(message => {
          if (!lastMessageId || message.id > lastMessageId) {
            addMessage(message)
            lastMessageId = message.id
          }
        })
      }
    } catch (error) {
      console.error('Error polling messages:', error)
    }
  }, 3000) // Poll every 3 seconds
}

function stopFallbackPolling() {
  if (fallbackPolling) {
    clearInterval(fallbackPolling)
    fallbackPolling = null
    console.log('📡 Stopped fallback polling')
  }
}

// Wait for service to be ready (handles Render cold starts)
async function waitForServiceReady(maxWaitMs = 15000) {
  const startTime = Date.now()
  const checkInterval = 1000 // Check every second

  while (Date.now() - startTime < maxWaitMs) {
    try {
      // Try hitting any API endpoint to see if service is awake
      const healthResp = await fetch(`${API_BASE_URL.replace('/api/v1', '')}/up`, {
        method: 'GET',
        signal: AbortSignal.timeout(2000) // 2 second timeout per check
      })

      if (healthResp.ok) {
        console.log('✅ Service is ready (woke up)')
        return true
      }
    } catch (e) {
      // Service not ready yet, continue waiting
      const elapsed = Math.round((Date.now() - startTime) / 1000)
      console.log(`⏳ Service waking up... (${elapsed}s)`)
    }

    // Wait before next check
    await new Promise(resolve => setTimeout(resolve, checkInterval))
  }

  console.warn('⚠️ Service wake-up check timeout - proceeding anyway (might be cold start)')
  return false
}

function scheduleReconnect() {
  if (!shouldReconnect || !currentRoom) return
  const delay = getReconnectDelay(reconnectAttempts)
  reconnectAttempts += 1
  console.log(`Scheduling reconnect attempt ${reconnectAttempts} in ${Math.round(delay)}ms`)
  reconnectTimeout = setTimeout(() => {
    if (shouldReconnect && reconnectAttempts < 10 && currentRoom) {
      // Clean up old connection
      if (ws) {
        try { ws.close() } catch (_) { }
        ws = null
      }
      connectWebSocket()
    } else {
      console.error('Max reconnection attempts reached')
      showError('Connection lost. Please refresh the page.')
    }
  }, delay)
}

async function connectWebSocket() {
  // Cancel any existing reconnection attempt
  if (reconnectTimeout) {
    clearTimeout(reconnectTimeout)
    reconnectTimeout = null
  }

  // If already connected, don't reconnect
  if (ws && ws.readyState === WebSocket.OPEN) {
    return
  }

  const token = await getStoredToken()
  if (!token) {
    showError('Authentication required')
    return
  }

  // Detect potential cold start without blocking the UI or connection attempt
  if (reconnectAttempts === 0) {
    console.log('🔍 Checking for cold start (non-blocking) ...')
    // Fire-and-forget: this will warm the service if sleepy but won’t delay WS
    waitForServiceReady(5000).catch(() => { })
  }

  // Fetch short-lived cable token for WS auth
  let cableToken
  try {
    const resp = await fetch(`${API_BASE_URL}/cable/token`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      signal: AbortSignal.timeout(5000) // 5 second timeout
    })
    if (!resp.ok) throw new Error(`Cable token error: ${resp.status}`)
    const data = await resp.json()
    cableToken = data.cable_token
  } catch (e) {
    console.error('Failed to fetch cable token:', e)
    // If token fetch fails, might be cold start - wait longer before retry
    if (reconnectAttempts === 0) {
      console.log('⏳ First connection failed - likely cold start, waiting 5s before retry')
      reconnectTimeout = setTimeout(() => {
        reconnectAttempts += 1
        connectWebSocket()
      }, 5000)
    } else {
      scheduleReconnect()
    }
    return
  }

  // Check WebSocket health before connecting (but don't block completely)
  try {
    console.log('🔍 Checking WebSocket health...')
    const healthResp = await fetch(`${API_BASE_URL}/websocket/health`, {
      signal: AbortSignal.timeout(3000) // 3 second timeout
    })
    if (healthResp.ok) {
      const healthData = await healthResp.json()
      console.log('WebSocket health check:', healthData)

      if (!healthData.redis_available) {
        console.error('❌ Redis not available - WebSocket will not work')
        // Don't block completely - fallback will handle it
      }
    }
  } catch (e) {
    console.warn('⚠️ WebSocket health check failed, proceeding anyway (might be cold start):', e.message)
  }

  const wsUrl = `${WS_URL}?cable_token=${encodeURIComponent(cableToken)}`

  try {
    ws = new WebSocket(wsUrl)

    ws.onopen = () => {
      console.log('✅ ActionCable WebSocket connected successfully')
      console.log('WebSocket readyState:', ws.readyState)
      console.log('WebSocket URL:', ws.url)
      reconnectAttempts = 0 // Reset on successful connection
      useFallback = false // Disable fallback when WebSocket works
      stopFallbackPolling() // Stop polling if it was running
      updateUserDisplay()
      // Wait a moment for connection to be fully established, then subscribe
      setTimeout(() => {
        console.log('Attempting to subscribe to room after WebSocket open...')
        subscribeToRoom()
      }, 200)
    }

    ws.onmessage = (event) => {
      try {
        const data = JSON.parse(event.data)
        console.log('WebSocket message received:', data)

        // Handle Action Cable welcome message
        if (data.type === 'welcome') {
          console.log('ActionCable welcome received')
          subscribeToRoom()
          return
        }

        // Handle subscription confirmation
        if (data.type === 'confirm_subscription') {
          console.log('Subscribed to room channel')
          isSubscribed = true
          // If we had enabled fallback, turn it off now
          useFallback = false
          stopFallbackPolling()
          updateUserDisplay()
          return
        }

        // Handle subscription rejection
        if (data.type === 'reject_subscription') {
          console.error('Subscription rejected - you are not a member of this room!')
          console.error('Rejection details:', data)
          isSubscribed = false

          // If subscription rejected, show join prompt
          if (currentRoom) {
            showJoinPrompt()
            showError('You must join the room to send messages')
          }
          updateUserDisplay()
          return
        }


        // Handle message broadcasts
        if (data.type === 'ping') {
          // Action Cable ping - ignore
          return
        }

        // Handle Action Cable message broadcasts - messages are wrapped in message.message
        if (data.message) {
          const messageData = data.message
          // Check if it's a message broadcast from our channel
          if (messageData.type === 'message' && messageData.message) {
            console.log('Received message:', messageData.message)
            addMessage(messageData.message)
            return
          }
        }

        // Handle actual message broadcasts from the channel (direct format)
        if (data.type === 'message' && data.message) {
          console.log('Received message:', data.message)
          addMessage(data.message)
          return
        }

        // Handle message deletions
        if (data.type === 'message_deleted' && data.message_id) {
          console.log('Message deleted via WebSocket:', data.message_id)
          removeMessage(data.message_id)
          return
        }

        // Fallback: check if it's already a message object
        if (data.content && data.user) {
          console.log('Received message (direct format):', data)
          addMessage(data)
        }
      } catch (error) {
        console.error('Error parsing WebSocket message:', error, event.data)
      }
    }

    ws.onclose = (event) => {
      const closeReason = {
        1000: 'Normal closure',
        1001: 'Going away',
        1002: 'Protocol error',
        1003: 'Unsupported data',
        1006: 'Abnormal closure (no close frame)',
        1007: 'Invalid data',
        1008: 'Policy violation',
        1009: 'Message too big',
        1010: 'Extension error',
        1011: 'Server error',
        1015: 'TLS handshake failure'
      }

      console.log('🔌 ActionCable WebSocket disconnected', {
        code: event.code,
        reason: event.reason || closeReason[event.code] || 'Unknown',
        wasClean: event.wasClean,
        shouldReconnect: shouldReconnect,
        hasRoom: !!currentRoom,
        isMember: currentRoom?.is_member,
        url: ws?.url
      })

      // Log specific error codes
      if (event.code === 1006) {
        console.error('🚨 WebSocket connection failed (1006) - Server may not be accepting WebSocket connections')
        console.error('💡 This often means:')
        console.error('   1. The /cable endpoint does not exist (404)')
        console.error('   2. The server does not support WebSockets')
        console.error('   3. There is a firewall/proxy blocking the connection')
      }

      isSubscribed = false
      updateUserDisplay()

      // Don't reconnect if we're not a member
      if (currentRoom && !currentRoom.is_member) {
        console.log('Not reconnecting - user is not a member')
        shouldReconnect = false
        return
      }

      // Only reconnect if it wasn't a clean close (code 1000) and we still want to reconnect
      // But stop reconnecting if we get 1006 repeatedly (connection failures)
      if (event.code === 1006 && reconnectAttempts >= 3) {
        console.error('🚨 Stopping reconnection attempts - WebSocket endpoint appears unavailable')
        console.log('📡 Switching to fallback polling mode...')
        shouldReconnect = false
        useFallback = true
        startFallbackPolling()
        updateUserDisplay()
        showError('Real-time chat unavailable. Using polling mode. Some features may be limited.')
        return
      }

      if (event.code !== 1000 && shouldReconnect && currentRoom && currentRoom.is_member) {
        scheduleReconnect()
      }
    }

    ws.onerror = (error) => {
      console.error('❌ ActionCable WebSocket error:', error)
      console.error('WebSocket error details:', {
        type: error.type,
        target: error.target,
        readyState: ws?.readyState,
        url: ws?.url,
        timestamp: new Date().toISOString()
      })

      // Try to get more details from the event
      if (error.target) {
        console.error('Error target URL:', error.target.url)
        console.error('Error target readyState:', error.target.readyState)
        console.error('Error target protocol:', error.target.protocol)
      }

      // Check if this is a connection failure
      if (ws?.readyState === 3) {
        console.error('🚨 WebSocket failed to connect - server may not support WebSockets or URL is incorrect')
      }

      // Error usually triggers onclose, so we'll handle reconnection there
      updateUserDisplay()
    }

    // Fast fallback: if we’re not subscribed shortly after attempting connection,
    // enable polling so the user can chat immediately while WS stabilizes.
    setTimeout(() => {
      if (!isSubscribed && currentRoom && currentRoom.is_member) {
        console.warn('⏱️ Fast fallback engaged: switching to polling while WS subscribes')
        useFallback = true
        startFallbackPolling()
        updateUserDisplay()
      }
    }, 1800)
  } catch (error) {
    console.error('Failed to create WebSocket connection:', error)
    scheduleReconnect()
  }
}

function subscribeToRoom() {
  if (ws && ws.readyState === WebSocket.OPEN && currentRoom) {
    console.log('Subscribing to room:', currentRoom.id)
    // Action Cable subscribe command format
    const subscribeMessage = {
      command: 'subscribe',
      identifier: JSON.stringify({
        channel: 'RoomChannel',
        room_id: currentRoom.id
      })
    }
    ws.send(JSON.stringify(subscribeMessage))
    console.log('Subscription message sent:', subscribeMessage)
  } else {
    console.error('Cannot subscribe - WebSocket not ready or no room', {
      wsReady: ws && ws.readyState === WebSocket.OPEN,
      hasRoom: !!currentRoom,
      wsState: ws ? ws.readyState : 'no ws'
    })
  }
}

function displayMessages(messages) {
  const messagesContainer = document.getElementById('messages')
  messagesContainer.innerHTML = ''

  messages.forEach(message => {
    addMessage(message)
  })
}

function addMessage(message) {
  const messagesContainer = document.getElementById('messages')

  // Check if message already exists (prevent duplicates)
  const messageId = message.id || `msg-${message.created_at}`
  const isTemp = String(messageId).startsWith('temp-')

  // If this is a real message (not temp), check for matching temp messages with same content
  if (!isTemp) {
    // Find and remove any temp messages with matching content + username
    const allMessages = messagesContainer.querySelectorAll('[data-message-id^="temp-"]')
    allMessages.forEach(tempEl => {
      const tempContent = tempEl.querySelector('.message-text')?.textContent || ''
      const tempUsername = tempEl.querySelector('.username')?.textContent || ''

      if (tempContent === message.content && tempUsername === (message.user?.username || 'Unknown')) {
        console.log('Removing temp message, real one arrived:', message.content)
        tempEl.remove()
      }
    })
  }

  // Check if this exact message ID already exists
  const existingMsg = messagesContainer.querySelector(`[data-message-id="${messageId}"]`)
  if (existingMsg) {
    // If it's a temp message, don't add duplicates
    if (isTemp) {
      return
    }
    // If it's a real message that already exists, remove the old one and replace it
    existingMsg.remove()
  }

  const messageEl = document.createElement('div')
  messageEl.className = 'message'
  messageEl.setAttribute('data-message-id', messageId)

  // Enhanced styling for temporary messages (gray/faded)
  if (isTemp) {
    messageEl.style.opacity = '0.5'
    messageEl.style.filter = 'grayscale(100%)'
  }

  const timestamp = new Date(message.created_at).toLocaleTimeString()

  messageEl.innerHTML = `
    <div class="avatar" style="${isTemp ? 'filter: grayscale(100%);' : ''}">${(message.user?.username || 'U').charAt(0).toUpperCase()}</div>
      <div class="message-content">
        <div class="message-header">
          <span class="username" style="${isTemp ? 'color: #6B7280;' : ''}">${escapeHtml(message.user?.username || 'Unknown')}</span>
          <span class="timestamp" style="${isTemp ? 'color: #9CA3AF;' : ''}">${timestamp}</span>
          ${isTemp ? '<span style="font-size: 11px; color: #F59E0B; font-style: italic; display: inline-flex; align-items: center; gap: 4px;"><span style="display: inline-block; width: 10px; height: 10px; border: 2px solid #F59E0B; border-top-color: transparent; border-radius: 50%; animation: spin 1s linear infinite;"></span>Sending...</span>' : ''}
          ${!isTemp && message.user?.id === currentUser?.id ? `<button class="delete-btn" data-message-id="${message.id}" style="opacity: 0; transition: opacity 0.2s; margin-left: 8px; background: none; border: none; color: #EF4444; cursor: pointer; font-size: 12px; padding: 2px 4px; border-radius: 3px;" title="Delete message">🗑️</button>` : ''}
        </div>
        <div class="message-text" style="${isTemp ? 'color: #6B7280;' : ''}">${escapeHtml(message.content)}</div>
      </div>
  `

  messagesContainer.appendChild(messageEl)
  messagesContainer.scrollTop = messagesContainer.scrollHeight

  // Add hover effect for delete button
  if (!isTemp && message.user?.id === currentUser?.id) {
    const deleteBtn = messageEl.querySelector('.delete-btn')
    if (deleteBtn) {
      messageEl.addEventListener('mouseenter', () => {
        deleteBtn.style.opacity = '1'
      })
      messageEl.addEventListener('mouseleave', () => {
        deleteBtn.style.opacity = '0'
      })

      deleteBtn.addEventListener('click', async (e) => {
        e.stopPropagation()
        if (confirm('Are you sure you want to delete this message?')) {
          try {
            const token = await getStoredToken()
            console.log('Attempting to delete message:', message.id)
            const response = await fetch(`${API_BASE_URL}/rooms/${currentRoom.id}/messages/${message.id}`, {
              method: 'DELETE',
              headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
              }
            })

            console.log('Delete response status:', response.status)
            if (response.ok) {
              console.log('Message deleted successfully from API')
              // Remove message from UI immediately since API call succeeded
              messageEl.remove()

              // Also set a fallback in case WebSocket broadcast fails
              setTimeout(() => {
                if (document.body.contains(messageEl)) {
                  console.log('WebSocket deletion broadcast failed, removing locally:', message.id)
                  messageEl.remove()
                }
              }, 3000)
            } else {
              const errorData = await response.json().catch(() => ({}))
              console.error('Delete failed:', errorData)
              showError(`Failed to delete message: ${errorData.error || response.statusText}`)
            }
          } catch (error) {
            console.error('Error deleting message:', error)
            showError('Failed to delete message. Please try again.')
          }
        }
      })
    }
  }
}

function removeMessage(messageId) {
  const messagesContainer = document.getElementById('messages')
  const messageEl = messagesContainer.querySelector(`[data-message-id="${messageId}"]`)
  if (messageEl) {
    // Add fade out animation
    messageEl.style.transition = 'opacity 0.3s ease-out, transform 0.3s ease-out'
    messageEl.style.opacity = '0'
    messageEl.style.transform = 'translateX(-20px)'

    setTimeout(() => {
      if (messageEl.parentNode) {
        messageEl.remove()
      }
    }, 300)
  }
}

function escapeHtml(text) {
  const div = document.createElement('div')
  div.textContent = text
  return div.innerHTML
}

function updateRoomInfo() {
  updateUserDisplay()
}

function showChat() {
  console.log('showChat() called - hiding join prompt and showing chat')
  document.getElementById('loading').style.display = 'none'
  document.getElementById('join-prompt').style.display = 'none'
  document.getElementById('chat-container').style.display = 'flex'
}

function showJoinPrompt() {
  console.log('showJoinPrompt() called - hiding chat and showing join prompt')
  document.getElementById('loading').style.display = 'none'
  document.getElementById('chat-container').style.display = 'none'
  document.getElementById('join-prompt').innerHTML = `
    <h3>Join Chat Room</h3>
    <p>There's already a conversation about this story!</p>
    <button id="join-button" class="join-button">Join Room</button>
  `
  document.getElementById('join-prompt').style.display = 'block'

  // Ensure button event listener is attached (in case it wasn't)
  const joinButton = document.getElementById('join-button')
  if (joinButton && !joinButton.hasAttribute('data-listener-attached')) {
    joinButton.setAttribute('data-listener-attached', 'true')
    joinButton.addEventListener('click', async (e) => {
      e.preventDefault()
      console.log('Join button clicked (from direct listener)')
      await joinRoom()
    })
  }
}

function showNoRoom() {
  document.getElementById('loading').style.display = 'none'
  document.getElementById('join-prompt').innerHTML = `
    <h3>No chat room for this page</h3>
    <p>Be the first to start a conversation about this story!</p>
    <button id="create-room-button" class="join-button">Create Room</button>
  `
  document.getElementById('join-prompt').style.display = 'block'
}

function showError(message) {
  document.getElementById('loading').style.display = 'none'
  document.getElementById('join-prompt').innerHTML = `
    <h3>Error</h3>
    <p>${message}</p>
  `
  document.getElementById('join-prompt').style.display = 'block'
}

function setupEventListeners() {
  // Message form submission
  document.getElementById('message-form').addEventListener('submit', async (e) => {
    e.preventDefault()
    const input = document.getElementById('message-input')
    const content = input.value.trim()

    // Try WebSocket first, fall back to REST API if needed
    if (content && ws && ws.readyState === WebSocket.OPEN && currentRoom && isSubscribed && !useFallback) {
      console.log('Sending message via WebSocket - all conditions met')
      // Action Cable perform command format
      const message = {
        command: 'message',
        identifier: JSON.stringify({
          channel: 'RoomChannel',
          room_id: currentRoom.id
        }),
        data: JSON.stringify({
          action: 'speak',
          message: content
        })
      }
      ws.send(JSON.stringify(message))
      input.value = ''
      console.log('Message sent via WebSocket:', content)

      // Optimistically add message to UI (will be replaced by server response)
      const optimisticMessage = {
        id: `temp-${Date.now()}`,
        content: content,
        user: { username: currentUser?.username || 'You' },
        created_at: new Date().toISOString()
      }
      addMessage(optimisticMessage)
    } else if (content && currentRoom && useFallback) {
      // Fallback: send via REST API
      console.log('Sending message via REST API (fallback mode)')
      try {
        const token = await getStoredToken()
        const response = await fetch(`${API_BASE_URL}/rooms/${currentRoom.id}/messages`, {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({ content: content })
        })

        if (response.ok) {
          const data = await response.json()
          input.value = ''
          console.log('Message sent via REST API:', content)

          // Add the message from server response (this will replace any temp message)
          if (data.message) {
            // Remove any matching temp messages first
            const messagesContainer = document.getElementById('messages')
            const allMessages = messagesContainer.querySelectorAll('[data-message-id^="temp-"]')
            allMessages.forEach(tempEl => {
              const tempContent = tempEl.querySelector('.message-text')?.textContent || ''
              const tempUsername = tempEl.querySelector('.username')?.textContent || ''

              if (tempContent === content && tempUsername === (currentUser?.username || 'You')) {
                tempEl.remove()
              }
            })

            addMessage(data.message)
            lastMessageId = data.message.id
          }
        } else {
          const errorData = await response.json().catch(() => ({}))
          console.error('Failed to send message:', response.status, errorData)
          showError(`Failed to send message: ${errorData.error || response.statusText}`)
        }
      } catch (error) {
        console.error('Error sending message via REST:', error)
        showError(`Failed to send message: ${error.message}`)
      }
    } else {
      console.error('Cannot send message - WebSocket not ready, no room, or not subscribed', {
        wsReady: ws && ws.readyState === WebSocket.OPEN,
        hasRoom: !!currentRoom,
        isSubscribed,
        wsState: ws ? ws.readyState : 'no ws'
      })

      // Show debug info to user
      const debugInfo = `Debug: WS=${ws?.readyState}, Room=${!!currentRoom}, Subscribed=${isSubscribed}`
      console.log(debugInfo)

      // Try to reconnect if not subscribed
      if (ws && ws.readyState === WebSocket.OPEN && currentRoom && !isSubscribed) {
        console.log('Attempting to resubscribe...')
        subscribeToRoom()
      }
    }
  })

  // Join room button
  document.addEventListener('click', async (e) => {
    if (e.target.id === 'join-button') {
      console.log('Join button clicked')
      await joinRoom()
    } else if (e.target.id === 'create-room-button') {
      console.log('Create room button clicked')
      await createRoom()
    }
  })
}

async function joinRoom() {
  try {
    const token = await getStoredToken()
    const response = await fetch(`${API_BASE_URL}/rooms/${currentRoom.id}/join`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    })

    if (response.ok) {
      // Refresh room data to get updated is_member status
      const refreshResponse = await fetch(`${API_BASE_URL}/rooms/search?url=${encodeURIComponent(currentRoom.canonical_url || '')}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      })

      if (refreshResponse.ok) {
        const refreshData = await refreshResponse.json()
        if (refreshData.room) {
          currentRoom = refreshData.room
          updateUserDisplay()
        }
      }

      // Update membership status immediately
      currentRoom.is_member = true
      shouldReconnect = true

      showChat()
      await loadMessages()
      // Wait a bit before connecting WebSocket to ensure room membership is updated
      setTimeout(() => {
        console.log('Connecting WebSocket after join...')
        connectWebSocket()
      }, 500)
    } else {
      const errorData = await response.json().catch(() => ({}))
      console.error('Failed to join room:', response.status, errorData)
      showError(`Failed to join room: ${errorData.error || response.statusText || 'Unknown error'}`)
    }
  } catch (error) {
    console.error('Error joining room:', error)
    showError(`Failed to join room: ${error.message}`)
  }
}

async function createRoom() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
    const token = await getStoredToken()

    const response = await fetch(`${API_BASE_URL}/rooms`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        url: tab.url,
        title: tab.title
      })
    })

    if (response.ok) {
      const data = await response.json()
      currentRoom = data.room
      updateRoomInfo()
      showChat()
      await loadMessages()
      connectWebSocket()
    } else {
      const errorData = await response.json().catch(() => ({}))
      console.error('Failed to create room:', response.status, errorData)
      showError(`Failed to create room: ${errorData.error || response.statusText || 'Unknown error'}`)
    }
  } catch (error) {
    console.error('Error creating room:', error)
    showError(`Failed to create room: ${error.message}`)
  }
}
